function [cubes] = cube_extract(stk,vertices,varargin)
%This function takes in an image stack and a set of vertices, and use the
%vertices as the central point to generate x by x by x cubes or image
%stacky goodness.
%Synatax: [cubes] = cube_extract(stk,vertices,'size',10,'z',0.7);
%Input:     stk = the image stack.  If a cell array of image stacks, the
%               cell must be structured as a horizontal vector, e.g.,
%               (1,x).  In the case of a cell array, the same cube will be
%               extracted from each cell field.
%           verticies = an array of x by 3, where each column is x y and z 
%               respectively.
%           'size' = the size of the cube.  Default = 10x10x10.  Note: the
%               acutal dimensions produced by the function with vertex is
%               11x11x11.
%           'x' = the scale, as a ratio or xyz, of 1 unit in the x axis.
%                 Default = 1
%           'y' = the scale, as a ratio or xyz, of 1 unit in the y axis.
%                 Default = 1
%           'z' = the scale, as a ratio or xyz, of 1 unit in the z axis.
%                 Default = 1
%Output:    cubes = the vertex centered cubes of image data.  Bitsize 
%               chunks of imaging goodness.

[xyz_pad] = parse(varargin);

%step through the channels or # of stacks entered.
cubes = [];     %initiate
if iscell(stk)      %cell processing
    parfor i = 1:size(stk,2)
        cubes_tmp = cube_out(stk{i,1},vertices,xyz_pad);
        if isempty(cubes)   %Like a virgin...
            cubes = cubes_tmp;
        else        %around and around...
            cubes = horzcat(cubes,cubes_tmp);
        end
    end
else                %one single stack.
    cubes = cube_out(stk,vertices,xyz_pad);
end

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [xyz_pad] = parse(input)

xyz_pad = 5;  %Default Initialized.
x_scale = 1;    %base scale of a unit in the x axis
y_scale = 1;    %base scale of a unit in the y axis
z_scale = 1;    %base scale of a unit in the z axis

%Parse the input
if ~isempty(input)
    for i = 1:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'size'
                    xyz_pad = round(input{1,i+1}/2);    %div by 2 for padding on each side of vertex.  Round for pixel coordinates.
                case 'x'
                    x_scale = input{1,i+1};
                case 'y'
                    y_scale = input{1,i+1};
                case 'z'
                    z_scale = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        end
    end
end
%lets make sure there are no subpixel declarations in x y or z
if x_scale~=1 || y_scale~=1 || z_scale~=1
    xyz = [x y z];      %put together for faster analysis
    xyz = ceil(xyz/max(xyz));   %put into pixel coordiantes and normalize to highest value which=1.
    x_scale = xyz(1,1);
    y_scale = xyz(1,2);
    z_scale = xyz(1,3);
end
    

%--------------------------------------------------------------------------
%subfunction to cube out the stack
function [cubes_tmp] = cube_out(stk,vertices,xyz_pad)

%get stack dimensions
[x,y,z] = size(stk);
xyz_dim = (xyz_pad*2)+1;    %the actual edges of the cube.  BTW don't ever ask me why I am doing this.
%Pad the array first so we won't get out of bounds
stk = padarray(stk,[xyz_pad xyz_pad]);      %pad x y
stk = cat(3,zeros(x,y,xyz_pad),stk,zeros(x,y,xyz_pad));
%lets make some cubes
%slice out the vertices
vert_x = vertices(:,1);
vert_y = vertices(:,2);
vert_z = vertices(:,3);
parfor i = 1:size(vertices,1)      %step through the vertices
    %x dimensions
    x_min = vert_x(i,1)-xyz_pad;
    %y second
    y_min = vert_y(i,1)-xyz_pad;
    %z third
    z_max = vert_z(i,1)+xyz_pad;
    z_min = vert_z(i,1)-xyz_pad;
    %now take out the cube
    stk_tmp = stk(:,:,z_min:z_max);     %set the z dimension.
    cubes_tmp{i,1} = stk_mod(stk,'crop',[xyz_dim xyz_dim],[x_min y_min]);
end